-- Usage:
-- Use this to get coordinates of units with minimal
-- interference with other addons.

-- If this changes map zoom it always changes it back to the original when
-- finished.

-- Use MapLibrary to detect if MapLibrary is running. It gets set on OnLoad.
-- Example: if MapLibrary then ... end

-- Functions:
-- GetMapZoneName - get the name of the zone that's currently on the map
-- GetWorldPosition - get the position of a unit in world coordinates
-- GetZonePosition - get the position of a unit in zone coordinates
-- TranslateWorldToZone - translate coordinates
-- TranslateZoneToWorld - translate coordinates
-- YardDistance - distance between two points in yards.
-- UnitDistance - distance between two units

local MapLibrary = nil;

-- returns the name of the currently shown map zone
-- returns nil of the currently shown map zone is not a real zone.
function GetMapZoneName()
   local continent = GetCurrentMapContinent();
   local zone = GetCurrentMapZone();

   if continent == 0 and zone == 0 then
      return "World";
   end

   local name = MapLibrary.zone_names[continent][zone];
   if MapLibrary.mappedzone[name] then
      return name;
   end

   return nil;
end

-- returns x, y relative to the world map on success.
-- returns nil on failure
-- if unit is nil then "player" is assumed
-- if corpse is nil then GetPlayerMapPosition is assumed,
-- otherwise GetPlayerCorpsePosition is used.
-- if nice is nil, then GetWorldPosition will actively try to
-- change map zoom to find the position!
function GetWorldPosition(unit, corpse, nice)
   local func = GetPlayerMapPosition
   if corpse then
      func = GetCorpseMapPosition
   end

   local zone = GetMapZoneName();
   if zone == nil then
      return;
   end

   if unit == nil then
      unit = "player";
   end

   -- First try to use the current zone.
   if ZoneIsCalibrated(zone) then
      local x, y = func(unit);
      if x ~= 0 or y ~= 0 then
	 return TranslateZoneToWorld(x, y, zone);
      end
   end

   if nice ~= nil then
      return nil;
   end

   local old_continent, old_zone = GetCurrentMapContinent(), GetCurrentMapZone();

   -- That failed. Zoom back to the zone we're in.
   local curzone = GetRealZoneText();
   if ZoneIsTranslated(curzone) then

      local id = MapLibrary.mappedzone[curzone];
      SetMapZoom(id.continent, id.zone);

      local x, y = func(unit);
      if x ~= 0 or y ~= 0 then
	 SetMapZoom(old_continent, old_zone);
	 return TranslateZoneToWorld(x, y, zone);
      end
      SetMapZoom(old_continent, old_zone);
      return nil;
   end

   -- That failed. Zoom back to the continent we're in.
   local continent = MapLibrary.continent_names[MapLibrary.mappedzone[cuzone].continent];

   if ZoneIsTranslated(continent) then
      local id = MapLibrary.mappedzone[continent];
      SetMapZoom(id.continent, id.zone);

      local x, y = func(unit);
      if x ~= 0 or y ~= 0 then
	 SetMapZoom(old_continent, old_zone);
	 return TranslateZoneToWorld(x, y, zone);
      end
      SetMapZoom(old_continent, old_zone);
      return nil;
   end

   -- Failed aswell. Zoom back to world.
   SetMapZoom(0, 0);

   local x, y = func(player);
   if x ~= 0 or y ~= 0 then
      SetMapZoom(old_continent, old_zone);
      return x, y
   end

   SetMapZoom(old_continent, old_zone);
   return nil;
end

-- returns x, y relative to the zone map on success.
-- see GetWorldPosition for details
function GetZonePosition(zone, unit, corpse, nice)
   if ZoneIsTranslated(zone) then
      local x, y = GetWorldPosition(unit, corpse, nice);
      return TranslateWorldToZone(x, y, zone);
   end
   return nil;
end

-- translates from world coordinates to zone coordinates.
-- returns nil on failure (if the zone is not calibrated)
function TranslateWorldToZone(wx, wy, zone)
   if ZoneIsCalibrated(zone) then
      local t = MapLibraryData.translation[zone];
      return t.offset_x + t.scale_x * wx, t.offset_y + t.scale_y * wy
   end
   return nil;
end

-- translates from zone coordinates to world coordinates.
-- returns nil on failure (if the zone is not calibrated)
function TranslateZoneToWorld(zx, zy, zone)
   if zone == "World" then
      return zx, zy;
   end

   if ZoneIsCalibrated(zone) then
      local t = MapLibraryData.translation[zone];
      return (zx - t.offset_x) / t.scale_x, (zy - t.offset_y) / t.scale_y;
   end
   return nil;
end

-- The parameters should be in world coordinates!
function YardDistance(x1, y1, x2, y2)
   local dx = (x1 - x2) * MapLibrary.yard_factor_x;
   local dy = (y1 - y2) * MapLibrary.yard_factor_y;
   return math.sqrt(dx * dx + dy * dy);
end

-- units can be "player", "party1", et.c.
-- set nice to 1 to avoid doing SetMapZoom
-- returns nil on failure
-- returns the distance in yards on success.
function UnitDistance(unit1, unit2, nice)
   local x1, y1 = GetWorldPosition(unit1, nil, nice);
   local x2, y2 = GetWorldPosition(unit2, nil, nice);
   if x1 == nil or x2 == nil then
      return nil;
   end
   return YardDistance(x1, y1, x2, y2);
end

----------------------------------------------------------------
-- Internal functions begin here!
-- These should not be called.
----------------------------------------------------------------

-- returns true if the zone is calibrated
-- returns false if the zone is not calibrated
-- returns nil if the zone can not be calibrated
function ZoneIsCalibrated(zone)

   if type(zone) ~= "string" then
      return nil;
   end

   if zone == "World" then
      return true;
   end

   local id = MapLibrary.mappedzone[zone];
   if not id then
      return nil;
   end

   if MapLibraryData.translation[zone].offset_x and
      MapLibraryData.translation[zone].offset_y then
      return true;
   else
      return false;
   end
end

function MapLibrary_OnLoad()
   -- This is for setting up persistent variable
   this:RegisterEvent("VARIABLES_LOADED");

   -- This is for calibrating zones
   this:RegisterEvent("PLAYER_ENTERING_WORLD");
   this:RegisterEvent("ZONE_CHANGED");
   this:RegisterEvent("ZONE_CHANGED_INDOORS");
   this:RegisterEvent("ZONE_CHANGED_NEW_AREA");

   MapLibrary = { };

   MapLibrary.continent_names = { GetMapContinents(); }
   MapLibrary.zone_names = { };

   MapLibrary.mappedzone = { };
   for i, cont_name in MapLibrary.continent_names do
      MapLibrary.zone_names[i] = { GetMapZones(i) };
      MapLibrary.zone_names[i][0] = cont_name;

      MapLibrary.mappedzone[cont_name] = {["continent"] = i, ["zone"] = 0};
      for index, name in MapLibrary.zone_names[i] do
	 MapLibrary.mappedzone[name] = {["continent"] = i, ["zone"] = index};
      end
   end

   -- These values are not perfect!
   MapLibrary.yard_factor_x = 100.0 / 0.00344;
   MapLibrary.yard_factor_y = 100.0 / 0.00344;
end

function MapLibrary_OnEvent()
   if event == "VARIABLES_LOADED" then
      if not MapLibraryData then
	 MapLibraryData = { };
      end

      if not MapLibraryData.translation then
	 MapLibraryData.translation = { };
      end

      for zone, tmp in MapLibrary.mappedzone do
	 if not MapLibraryData.translation[zone] then
	    MapLibraryData.translation[zone] = { };
	 end
      end
   elseif event == "PLAYER_ENTERING_WORLD" or
      event == "ZONE_CHANGED" or
      event == "ZONE_CHANGED_NEW_AREA" or
      event == "ZONE_CHANGE_INDOORS" then
      
      local zone = GetRealZoneText();

      if ZoneIsCalibrated(zone) == false then
	 MapLibrary_Updater:Show();
      end
   end
end

local MapLibrary_LastUpdate = 0;

function MapLibrary_OnUpdate()
   -- don't do it too often!
   local time = GetTime();
   if time - MapLibrary_LastUpdate > 1 then
      MapLibrary_LastUpdate = time;
   else
      return;
   end

   local zone = GetRealZoneText();
   local continent = nil;

   local b = ZoneIsCalibrated(zone);
   if b == nil then
      MapLibrary_Updater:Hide();
   else
      continent = MapLibrary.continent_names[MapLibrary.mappedzone[zone].continent];
      local c = ZoneIsCalibrated(continent)

      if (b == true) and (c == true) then
	 MapLibrary_Updater:Hide();
      else
	 if b == false then
	    MapLibrary_CalculateCurrentTranslation(zone);
	 end
	 if c == false then
	    MapLibrary_CalculateCurrentTranslation(continent);
	 end
      end
   end
end

function MapLibrary_CalculateCurrentTranslation(zone)
   local b = ZoneIsCalibrated(zone);
   if b == nil or b == true then
      return;
   end

   local id = MapLibrary.mappedzone[zone];

   local old_continent, old_zone = GetCurrentMapContinent(), GetCurrentMapZone();
   -- Get map coordinates
   SetMapZoom(0, 0);
   local wx, wy = GetPlayerMapPosition("player");
   if wx == 0 and wy == 0 then
      SetMapZoom(old_continent, old_zone);
      return;
   end

   SetMapZoom(id.continent, id.zone);

   local zx, zy = GetPlayerMapPosition("player");
   if zx == 0 and zy == 0 then
      SetMapZoom(old_continent, old_zone);
      return;
   end

   -- Calibrate x and y separately, for more efficiency.

   -- If this is the first calibration run of this zone:
   if MapLibraryData.translation[zone].min_wx == nil then
      MapLibraryData.translation[zone].min_wx = wx;
      MapLibraryData.translation[zone].min_zx = zx;

      MapLibraryData.translation[zone].max_wx = wx;
      MapLibraryData.translation[zone].max_zx = zx;
   end

   if MapLibraryData.translation[zone].min_wy == nil then
      MapLibraryData.translation[zone].min_wy = wy;
      MapLibraryData.translation[zone].min_zy = zy;

      MapLibraryData.translation[zone].max_wy = wy;
      MapLibraryData.translation[zone].max_zy = zy;
   end

   -- Update min and max values
   if wx < MapLibraryData.translation[zone].min_wx then
      MapLibraryData.translation[zone].min_wx = wx;
      MapLibraryData.translation[zone].min_zx = zx;
   end
   if wx > MapLibraryData.translation[zone].max_wx then
      MapLibraryData.translation[zone].max_wx = wx;
      MapLibraryData.translation[zone].max_zx = zx;
   end
   if wy < MapLibraryData.translation[zone].min_wy then
      MapLibraryData.translation[zone].min_wy = wy;
      MapLibraryData.translation[zone].min_zy = zy;
   end
   if wy > MapLibraryData.translation[zone].max_wy then
      MapLibraryData.translation[zone].max_wy = wy;
      MapLibraryData.translation[zone].max_zy = zy;
   end

   -- Check if the distance is enough for calibration:
   local dx = MapLibraryData.translation[zone].max_wx - MapLibraryData.translation[zone].min_wx;
   local dy = MapLibraryData.translation[zone].max_wy - MapLibraryData.translation[zone].min_wy;

   if dx * MapLibrary.yard_factor_x > 5 then
      local min_wx = MapLibraryData.translation[zone].min_wx;
      local max_wx = MapLibraryData.translation[zone].max_wx;

      local min_zx = MapLibraryData.translation[zone].min_zx;
      local max_zx = MapLibraryData.translation[zone].max_zx;

      local scale_x = (max_zx - min_zx) / (max_wx - min_wx);
      MapLibraryData.translation[zone].scale_x = scale_x;
      MapLibraryData.translation[zone].offset_x = max_zx - scale_x * max_wx;
   end

   if dy * MapLibrary.yard_factor_y > 5 then
      local min_wy = MapLibraryData.translation[zone].min_wy;
      local max_wy = MapLibraryData.translation[zone].max_wy;

      local min_zy = MapLibraryData.translation[zone].min_zy;
      local max_zy = MapLibraryData.translation[zone].max_zy;

      local scale_y = (max_zy - min_zy) / (max_wy - min_wy);
      MapLibraryData.translation[zone].scale_y = scale_y;
      MapLibraryData.translation[zone].offset_y = max_zy - scale_y * max_wy;
   end

   if dx * MapLibrary.yard_factor_x > 5 and
      dy * MapLibrary.yard_factor_y > 5 then

      -- Don't need to calibrate anymore, so remove temporary data
      MapLibraryData.translation[zone].min_wx = nil;
      MapLibraryData.translation[zone].min_wy = nil;
      MapLibraryData.translation[zone].max_wx = nil;
      MapLibraryData.translation[zone].max_wy = nil;

      MapLibraryData.translation[zone].min_zx = nil;
      MapLibraryData.translation[zone].min_zy = nil;
      MapLibraryData.translation[zone].max_zx = nil;
      MapLibraryData.translation[zone].max_zy = nil;

      MapLibrary_Msg("Calibrated " .. zone);
   end
   SetMapZoom(old_continent, old_zone);
end

function MapLibrary_Msg(s)
   if(DEFAULT_CHAT_FRAME) then
      DEFAULT_CHAT_FRAME:AddMessage("MapLibrary: " .. s);
   end
end
