--[[
 Chat Filter v0.21
    By Anders "sarf" Kronquist
  
  Replaces certain patterns in text with other patterns.

  Written to allow people to control their own perception of the world,
   but it can be circumvented. Basically, rose-tinted glasses for your chat-box,
   that will not prevent someone to thrown garbage in your face.
  
  Note that this Add-On does not require Cosmos, but if you do not have Cosmos 
   you will have to change the ChatFilter_X variables yourself.
  
  I also humbly admit that I based this file on the CombatCaller.lua file,
   and got much inspiration from the rest of Cosmos.

History
-------

v0.21
	May have fixed a small bug related to options not saving/being enabled by default.
	Fixed the Interface version number

v0.2
	Fixed a bug which meant that no incoming messages would get filtered.

v0.1
	Created the Chat Filter.

  ]]--

-- Constants

-- The maximum number many times the filter loop is allowed to be run.
MAX_FILTERING_LOOPS = 5;

-- These will be the values with checkboxes
ChatFilter_Enabled = false;
ChatFilter_Incoming_Filter_Enabled = false;
ChatFilter_Outgoing_Filter_Enabled = false;
ChatFilter_Whisper_Enabled = false;
ChatFilter_Filter_Profanity = false;
ChatFilter_Filter_Leet = false;
ChatFilter_Filter_User = false;

-- These will be the variable variables. ;)
ChatFilter_Filters = { };
ChatFilter_ChatTypesToFilter = { };

-- SETUP FUNCTIONS

-- retrieves the chat types that should be filtered
function ChatFilter_GetChatTypesToFilter(addWhisperTypes)
	chatTypes = {
-- say types
		"CHAT_MSG_SAY",
		"CHAT_MSG_EMOTE",
		"CHAT_MSG_TEXT_EMOTE",
-- yell types
		"CHAT_MSG_YELL",
-- party types
		"CHAT_MSG_PARTY",
-- guild types
		"CHAT_MSG_GUILD",
		"CHAT_MSG_OFFICER",
		"GUILD_MOTD",
-- monster types
		"CHAT_MSG_MONSTER_SAY",
		"CHAT_MSG_MONSTER_YELL",
		"CHAT_MSG_MONSTER_EMOTE",
		"CHAT_MSG_MONSTER_WHISPER",
	};
	if(addWhisperTypes) then
		chatTypes[table.getn(chatTypes)+1] = "CHAT_MSG_WHISPER";
		chatTypes[table.getn(chatTypes)+1] = "CHAT_MSG_WHISPER_INFORM";
	end
	return chatTypes;
end

-- set up the chat types to filter
function ChatFilter_SetupChatTypesToFilter()
{
	ChatFilter_Filters = ChatFilter_GetChatTypesToFilter(ChatFilter_Whisper_Enabled);
}

-- should a incoming message with the specified event be filtered?
function ChatFilter_ShouldFilterEvent(event)
	if((ChatFilter_Incoming_Filter_Enabled == nil) or (ChatFilter_Incoming_Filter_Enabled == false)) then
		return false;
	end
	for key,value in ChatFilter_ChatTypesToFilter do
		if(event == value) then
			return true;
		end
	end
	return false;
end

-- add a filter-array to the main filter
function ChatFilter_AddFilters(filters)
	for key, value in filters
		ChatFilter_Filters[key] = value;
	end
end

-- update the main filter
function ChatFilter_UpdateFilters()
	ChatFilter_Filters = {};
	if(ChatFilter_Filter_Profanity) then
		ChatFilter_AddFilters(CHAT_FILTER_PROFANITY_LIST);
	end
	if(ChatFilter_Filter_Leet) then
		ChatFilter_AddFilters(CHAT_FILTER_LEET_LIST);
	end
	if(ChatFilter_Filter_User) then
		ChatFilter_AddFilters(CHAT_FILTER_USER_LIST);
	end
end

-- filter a message - returns the filtered message
function ChatFilter_FilterMessage(message)
	local newmessage = message;
	local hasFiltered = 1;
	local loops = 0;
	repeat 
		hasFiltered = 0;
		for key, value in ChatFilter_Filters
			-- TODO: do permutations (case-independent?) matching/replacing here
			newmessage, instances = gsub(newmessage, key, value);
			if ( instances > 0 ) then
				hasFiltered = 1;
			end
		end
		loops = loops + 1;
	until ((hasFiltered == 0) or (loops >= MAX_FILTERING_LOOPS))
	return newmessage;
end

-- should the outgoing message be filtered?
function ChatFilter_ShouldFilterOutgoingMessage(editBox)
	if((ChatFilter_Outgoing_Filter_Enabled == nil) or (ChatFilter_Outgoing_Filter_Enabled == false)) then
		return false;
	else
		local type = editBox.chatType;
		if((type ~= "WHISPER") or ((ChatFilter_Whisper_Enabled == true) or (ChatFilter_Whisper_Enabled == 1))) then
			return true;
		end
	end
end

-- filter the outgoing message
function ChatFilter_FilterOutgoingMessage(editBox)
	local message = editBox:GetText();
	editBox:SetText(ChatFilter_FilterMessage(message));
end

-- main set-up function
function ChatFilter_OnLoad() 

	ChatFilter_SetupChatTypesToFilter();

	ChatFilter_ChatFrame_OnEvent = ChatFrame_OnEvent;
	function ChatFrame_OnEvent(event, message)
		if(ChatFilter_ShouldFilterEvent(event)) then
			message = ChatFilter_FilterMessage(message);
		end
		ChatFilter_ChatFrame_OnEvent(event, message, arg);
	end
	
	ChatFilter_ChatEdit_SendText = ChatEdit_SendText;
	function ChatEdit_SendText(editBox, addHistory)
		if(ChatFilter_ShouldFilterOutgoingMessage(editBox)) then
			ChatFilter_FilterOutgoingMessage(editBox);
		end
		ChatFilter_ChatEdit_SendText(editBox, addHistory, arg);
	end

	if(CosmosMaster_Register) then
		ChatFilter_OnLoad_Cosmos();
	end

end

-- Cosmos setup-function
function ChatFilter_OnLoad_Cosmos()
	-- Register with the CosmosMaster
	CosmosMaster_Register(
		"COS_CHATF_HEADER",
		"SEPARATOR",
		CHATF_HEADER,
		CHATF_HEADER_INFO
	);
	CosmosMaster_Register("COS_CHATF_ENABLE_INCOMING",
		"CHECKBOX",
		CHATF_ENABLE_INCOMING,
		CHATF_ENABLE_INCOMING_INFO,
		ChatFilter_HandleEnableIncomingChange,
		0,
		0
	);
	CosmosMaster_Register("COS_CHATF_ENABLE_OUTGOING",
		"CHECKBOX",
		CHATF_ENABLE_OUTGOING,
		CHATF_ENABLE_OUTGOING_INFO,
		ChatFilter_HandleEnableOutgoingChange,
		0,
		0
	);
	CosmosMaster_Register("COS_CHATF_ENABLE_WHISPER",
		"CHECKBOX",
		CHATF_ENABLE_WHISPER,
		CHATF_ENABLE_WHISPER_INFO,
		ChatFilter_HandleEnableWhisperChange,
		0,
		0
	);

	CosmosMaster_Register("COS_CHATF_FILTER_PROFANITY",
		"CHECKBOX",
		CHATF_FILTER_PROFANITY,
		CHATF_FILTER_PROFANITY_INFO,
		ChatFilter_HandleFilterProfanityChange,
		0,
		0
	);
	CosmosMaster_Register("COS_CHATF_FILTER_LEET",
		"CHECKBOX",
		CHATF_FILTER_LEET,
		CHATF_FILTER_LEET_INFO,
		ChatFilter_HandleFilterLeetChange,
		0,
		0
	);
	CosmosMaster_Register("COS_CHATF_FILTER_USER",
		"CHECKBOX",
		CHATF_FILTER_USER,
		CHATF_FILTER_USER_INFO,
		ChatFilter_HandleFilterUserChange,
		0,
		0
	);
	
	-- TODO: add slash-commands
end

-- handles a change of whether incoming messages should be filtered
function ChatFilter_HandleEnableIncomingChange(checked, value)
	ChatFilter_Incoming_Filter_Enabled = (checked == 1);
end	

-- handles a change of whether outgoing messages should be filtered
function ChatFilter_HandleEnableOutgoingChange(checked, value)
	ChatFilter_Outgoing_Filter_Enabled = (checked == 1);
end	

-- handles a change of whether whispers should be filtered
function ChatFilter_HandleEnableWhisperChange(checked, value)
	ChatFilter_Whisper_Enabled = (checked == 1);
	ChatFilter_SetupChatTypesToFilter();
end	

-- handles a change of whether profanity should be filtered
function ChatFilter_HandleFilterProfanityChange(checked, value)
	ChatFilter_Filter_Profanity = (checked == 1);
	ChatFilter_UpdateFilters();
end	

-- handles a change of whether "leet speak" should be filtered
function ChatFilter_HandleFilterLeetChange(checked, value)
	ChatFilter_Filter_Leet = (checked == 1);
	ChatFilter_UpdateFilters();
end	

-- handles a change of whether user specified words should be filtered
function ChatFilter_HandleFilterUserChange(checked, value)
	ChatFilter_Filter_User = (checked == 1);
	ChatFilter_UpdateFilters();
end	


